<?php
require_once 'db.php';

/** ارسال پیام به کاربر */
function sendMessage($chat_id, $text, $reply_markup = null) {
    $url = "https://api.telegram.org/bot".BOT_TOKEN."/sendMessage";
    $data = [
        'chat_id' => $chat_id,
        'text' => $text,
        'parse_mode' => 'HTML'
    ];
    if ($reply_markup) {
        $data['reply_markup'] = json_encode($reply_markup, JSON_UNESCAPED_UNICODE);
    }
    file_get_contents($url."?".http_build_query($data));
}

/** گرفتن محصولات از API */
function getProducts($page = 1, $limit = 10, $filters = []) {
    $url = API_BASE."Search/Products?page=$page&limit=$limit";
    $opts = [
        'http' => [
            'method'  => 'POST',
            'header'  => 'Content-Type: application/json',
            'content' => json_encode((object)$filters, JSON_UNESCAPED_UNICODE)
        ]
    ];
    $context = stream_context_create($opts);
    return json_decode(file_get_contents($url, false, $context), true);
}

/** گرفتن اطلاعات هر محصول (ویژگی‌ها مثل رنگ و سایز) */
function getProductDetails($id) {
    $url = API_BASE."Variations/$id";
    return json_decode(file_get_contents($url), true)['Result'] ?? null;
}

/** گرفتن نام و جزئیات محصول با آیدی */
function getProductById($id) {
    $url = API_BASE . "Products/$id";
    $response = file_get_contents($url);
    $data = json_decode($response, true);
    return ($data && $data['Code'] == 1) ? $data['Result'] : null;
}

/** مدیریت کاربر */
function getUser($telegram_id) {
    global $db;
    $stmt = $db->prepare("SELECT * FROM users WHERE telegram_id = ?");
    $stmt->execute([$telegram_id]);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}
function upsertUserField($telegram_id, $field, $value) {
    global $db;
    $allowed = ['name','phone','address','postal_code'];
    if (!in_array($field, $allowed)) return;
    $stmt = $db->prepare("UPDATE users SET $field = ? WHERE telegram_id = ?");
    $stmt->execute([trim($value), $telegram_id]);
}
function completeUser($telegram_id) {
    global $db;
    $stmt = $db->prepare("UPDATE users SET is_completed = 1 WHERE telegram_id = ?");
    $stmt->execute([$telegram_id]);
}
function isUserCompleted($telegram_id) {
    global $db;
    $stmt = $db->prepare("SELECT is_completed FROM users WHERE telegram_id = ?");
    $stmt->execute([$telegram_id]);
    return (bool)$stmt->fetchColumn();
}
function getUserInfo($telegram_id) {
    global $db;
    $stmt = $db->prepare("SELECT * FROM users WHERE telegram_id = ?");
    $stmt->execute([$telegram_id]);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

/** مدیریت وضعیت (user_states) */
function getUserState($telegram_id) {
    global $db;
    $stmt = $db->prepare("SELECT * FROM user_states WHERE telegram_id = ?");
    $stmt->execute([$telegram_id]);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}
function setUserState($telegram_id, $state, $product_id = null, $colors = null) {
    global $db;
    $stmt = $db->prepare("
        INSERT INTO user_states (telegram_id, current_state, last_product_id, temp_colors)
        VALUES (?, ?, ?, ?)
        ON DUPLICATE KEY UPDATE 
            current_state=VALUES(current_state),
            last_product_id=VALUES(last_product_id),
            temp_colors=VALUES(temp_colors)
    ");
    $stmt->execute([$telegram_id, $state, $product_id, $colors]);
}
function updateUserStateField($telegram_id, $field, $value) {
    global $db;
    $allowed = ['selected_color', 'selected_size'];
    if (!in_array($field, $allowed)) return;
    $stmt = $db->prepare("UPDATE user_states SET $field = ? WHERE telegram_id = ?");
    $stmt->execute([$value, $telegram_id]);
}
function clearUserState($telegram_id) {
    global $db;
    $stmt = $db->prepare("DELETE FROM user_states WHERE telegram_id = ?");
    $stmt->execute([$telegram_id]);
}

/** سبد خرید */
function addToCart($telegram_id, $product_id, $variation_id, $color, $size, $quantity = 1, $unit_price) {
    global $db;

    $unit_price = (int)$unit_price;
    $quantity   = max(1, (int)$quantity);

    try {
        // چک وجود آیتم مشابه
        $stmt = $db->prepare("
            SELECT id, quantity, price FROM cart 
            WHERE telegram_id = ? AND product_id = ? AND variation_id = ? AND color = ? AND size = ?
            LIMIT 1
        ");
        if (!$stmt->execute([$telegram_id, $product_id, $variation_id, $color, $size])) {
            logToFile(['db_error_select' => $stmt->errorInfo()]);
            return false;
        }
        $existing = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($existing) {
            $new_quantity = (int)$existing['quantity'] + $quantity;

            if ($unit_price > 0 && $unit_price !== (int)$existing['price']) {
                $stmt = $db->prepare("UPDATE cart SET quantity = ?, price = ?, created_at = NOW() WHERE id = ?");
                if (!$stmt->execute([$new_quantity, $unit_price, $existing['id']])) {
                    logToFile(['db_error_update' => $stmt->errorInfo()]);
                    return false;
                }
            } else {
                $stmt = $db->prepare("UPDATE cart SET quantity = ?, created_at = NOW() WHERE id = ?");
                if (!$stmt->execute([$new_quantity, $existing['id']])) {
                    logToFile(['db_error_update_no_price' => $stmt->errorInfo()]);
                    return false;
                }
            }
        } else {
            $stmt = $db->prepare("
                INSERT INTO cart (telegram_id, product_id, variation_id, color, size, quantity, price, created_at)
                VALUES (?, ?, ?, ?, ?, ?, ?, NOW())
            ");
            if (!$stmt->execute([$telegram_id, $product_id, $variation_id, $color, $size, $quantity, $unit_price])) {
                logToFile(['db_error_insert' => $stmt->errorInfo()]);
                return false;
            }
        }

        return true;

    } catch (Throwable $e) {
        logToFile(['exception' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
        return false;
    }
}


/** گرفتن محصولات سبد خرید با قیمت واقعی از API */
function getCartItems($telegram_id) {
    global $db;
    $stmt = $db->prepare("
        SELECT id, product_id, variation_id, color, size, quantity 
        FROM cart
        WHERE telegram_id = ?
    ");
    $stmt->execute([$telegram_id]);
    $items = $stmt->fetchAll(PDO::FETCH_ASSOC);

    foreach ($items as &$item) {
        $productInfo = getProductById($item['product_id']);
        $variations = getProductDetails($item['product_id']);
        $variationPrice = 0;

        if ($variations) {
            foreach ($variations as $v) {
                if ($v['Id'] == $item['variation_id']) {
                    $variationPrice = $v['DiscountPrice'] > 0 ? $v['DiscountPrice'] : $v['Price'];
                    break;
                }
            }
        }

        $item['product_name'] = $productInfo ? $productInfo['Name'] : 'محصول';
        $item['price'] = $variationPrice ?: 0;
    }

    return $items;
}

/** تمیز کردن متن توضیحات */
function cleanDescription($text) {
    $text = preg_replace("/(\n\s*){3,}/u", "\n\n", $text);
    return trim($text);
}

/** جستجو بر اساس نام */
function searchProductsByName($query, $page = 1, $limit = 10) {
    $url = API_BASE . "Search/Products?page=$page&limit=$limit";
    $filters = ['Name' => $query];
    $opts = [
        'http' => [
            'method'  => 'POST',
            'header'  => 'Content-Type: application/json',
            'content' => json_encode($filters, JSON_UNESCAPED_UNICODE)
        ]
    ];
    $context = stream_context_create($opts);
    return json_decode(file_get_contents($url, false, $context), true);
}

/** ابزارهای متفرقه */
function normalizeText($text) {
    $text = trim($text);
    $text = preg_replace('/\x{200C}/u', ' ', $text);
    $text = str_replace(['ي','ك'], ['ی','ک'], $text);
    $text = preg_replace('/\s+/u', ' ', $text);
    return $text;
}
function findVariation($variations, $color, $size) {
    $color = normalizeText($color);
    $size = normalizeText($size);

    foreach ($variations as $v) {
        $vColor = normalizeText($v['Color']);
        $vSizes = array_map('normalizeText', $v['Sizes']);

        $hasColor = !empty($vColor);
        $hasSize  = !empty($vSizes);

        $match = false;

        if ($hasColor && $hasSize) {
            // رنگ و سایز هر دو باید تطبیق داشته باشند
            $match = ($vColor === $color && in_array($size, $vSizes));
        } elseif ($hasColor && !$hasSize) {
            // فقط رنگ چک میشه
            $match = ($vColor === $color);
        } elseif (!$hasColor && $hasSize) {
            // فقط سایز چک میشه
            $match = (in_array($size, $vSizes));
        } else {
            // نه رنگ داره نه سایز → اولینش رو انتخاب کن
            $match = true;
        }

        if ($match) {
            $price    = isset($v['Price']) ? (int)$v['Price'] : 0;
            $discount = isset($v['DiscountPrice']) ? (int)$v['DiscountPrice'] : 0;
            $final    = ($discount > 0) ? $discount : $price;

            $v['price']    = $price;
            $v['discount'] = $discount;
            $v['final']    = $final;
            return $v;
        }
    }
    return null;
}

function deleteMessage($chat_id, $message_id) {
    $url = "https://api.telegram.org/bot".BOT_TOKEN."/deleteMessage";
    $data = ['chat_id' => $chat_id, 'message_id' => $message_id];
    file_get_contents($url."?".http_build_query($data));
}
function logToFile($data) {
    file_put_contents(__DIR__.'/zarinpal_response.txt', date('Y-m-d H:i:s') . ' - ' . print_r($data, true) . "\n", FILE_APPEND);
}

/** منوها */
// منوهای اصلی
function getMainMenu() {
    return [
        'keyboard' => [
            ['🛒 سبد خرید', '👕 دسته بندی محصولات'],  // تغییر "لیست محصولات" به "دسته بندی محصولات"
            ['🔍 جستجوی محصول', '📦 پیگیری سفارشات', '👤 مشخصات من']
        ],
        'resize_keyboard' => true
    ];
}


function getCartMenu() {
    return [
        'keyboard' => [
            ['🗑 حذف همه محصولات', '💳 پرداخت'],
            ['⬅️ بازگشت به منو اصلی']
        ],
        'resize_keyboard' => true
    ];
}
function getCancelMenu() {
    return [
        'keyboard' => [
            ['انصراف']
        ],
        'resize_keyboard' => true
    ];
}
// تابع تبدیل اعداد فارسی به انگلیسی
function faToEnNumbers($string) {
    $persian = ['۰','۱','۲','۳','۴','۵','۶','۷','۸','۹'];
    $english = ['0','1','2','3','4','5','6','7','8','9'];
    return str_replace($persian, $english, $string);
}
// درخواست ایجاد تراکنش زرین‌پال
function zarinpal_request($merchant_id, $amount, $description, $callback_url) {
    logToFile("Starting Zarinpal request with Merchant ID: $merchant_id"); // لاگ کردن شروع درخواست

    $data = [
        "merchant_id" => $merchant_id,
        "amount"      => $amount,
        "currency"    => "IRT",
        "callback_url"=> $callback_url,
        "description" => $description
    ];

    logToFile("Zarinpal request data: " . print_r($data, true)); // لاگ کردن داده‌های ارسالی به زرین‌پال

    $ch = curl_init("https://api.zarinpal.com/pg/v4/payment/request.json");
    curl_setopt($ch, CURLOPT_USERAGENT, "ZarinPal Rest Api v4");
    curl_setopt($ch, CURLOPT_CUSTOMREQUEST, "POST");
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, ["Content-Type: application/json"]);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);

    $result = curl_exec($ch);

    if ($result === false) {
        logToFile("cURL error: " . curl_error($ch)); // ذخیره خطای cURL در صورت بروز مشکل در ارسال درخواست
        curl_close($ch);
        return false;
    }

    curl_close($ch);

    logToFile("Zarinpal API response: " . $result); // ذخیره پاسخ دریافتی از زرین‌پال

    $resultData = json_decode($result, true);
    if (isset($resultData['data']['code']) && $resultData['data']['code'] == 100) {
        logToFile("Zarinpal success: " . print_r($resultData, true)); // ذخیره موفقیت درخواست
        return [
            'authority' => $resultData['data']['authority'],
            'link'      => "https://www.zarinpal.com/pg/StartPay/" . $resultData['data']['authority']
        ];
    } else {
        logToFile("Zarinpal error: " . print_r($resultData, true)); // ذخیره خطا در صورت بروز مشکل از سمت زرین‌پال
        return false;
    }
}



// تأیید پرداخت در Callback
function zarinpal_verify($merchant_id, $amount, $authority) {
    $data = [
        "MerchantID" => $merchant_id,
        "Amount"     => $amount,
        "Authority"  => $authority
    ];

    $ch = curl_init('https://www.zarinpal.com/pg/rest/WebGate/PaymentVerification.json');
    curl_setopt($ch, CURLOPT_USERAGENT, 'ZarinPal Rest Api v1');
    curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'POST');
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'Content-Length: ' . strlen(json_encode($data))
    ]);

    $result = curl_exec($ch);
    $err = curl_error($ch);
    curl_close($ch);

    if ($err) {
        logToFile(['zarinpal_verify_error' => $err]);
        return false;
    } else {
        return json_decode($result, true);
    }
}

function sendUserOrders($chat_id) {
    global $db;

    $stmt = $db->prepare("SELECT * FROM orders WHERE telegram_id = ? ORDER BY id DESC");
    $stmt->execute([$chat_id]);
    $orders = $stmt->fetchAll(PDO::FETCH_ASSOC);

    if (!$orders) {
        sendMessage($chat_id, "📦 شما هیچ سفارشی ثبت نکرده‌اید.");
        return;
    }

    foreach ($orders as $order) {
        $user = json_decode($order['user_snapshot'], true);
        $date = $order['paid_at'] ?: $order['created_at'];
        $amount = number_format($order['amount']) . " تومان";

        $text = "🆔 شماره سفارش: #{$order['id']}\n";
        $text .= "📅 تاریخ: {$date}\n";
        $text .= "💰 مبلغ: {$amount}";

        $inlineKeyboard = [
            'inline_keyboard' => [
                [
                    ['text' => "🔍 مشاهده جزئیات", 'callback_data' => "order_details_{$order['id']}"]
                ]
            ]
        ];

        sendMessage($chat_id, $text, $inlineKeyboard);
    }
}
function sendOrderDetails($chat_id, $orderId) {
    global $db;

    $stmt = $db->prepare("SELECT * FROM orders WHERE id = ? AND telegram_id = ?");
    $stmt->execute([$orderId, $chat_id]);
    $order = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$order) {
        sendMessage($chat_id, "❌ سفارش یافت نشد.");
        return;
    }

    $itemsStmt = $db->prepare("SELECT * FROM order_items WHERE order_id = ?");
    $itemsStmt->execute([$orderId]);
    $items = $itemsStmt->fetchAll(PDO::FETCH_ASSOC);

    $user = json_decode($order['user_snapshot'], true);

    $msg = "شماره سفارش: {$orderId}\n";
    $msg .= "<b>تاریخ:</b> {$order['paid_at']}\n";
    $msg .= "<b>مبلغ:</b> " . number_format($order['amount']) . " تومان\n\n";
    
    $msg .= "🛍 <b>محصولات:</b>\n";
    foreach ($items as $it) {
        $msg .= "• {$it['product_name']} ({$it['color']} / {$it['size']}) × {$it['quantity']}\n";
    }
    
    $msg .= "\nسفارش شما <b>{$order['process_status']}</b> است.";





    sendMessage($chat_id, $msg, null, 'Markdown');
}

function decreaseStock($variationId, $quantity) {
    $url = API_BASE . "Bot/Stock/" . $variationId;
    
    $opts = [
        'http' => [
            'method'  => 'POST',
            'header'  => "Content-Type: application/json\r\nX-Bot-Token: " . BOT_STOCK_TOKEN,
            'content' => json_encode($quantity)
        ]
    ];
    $context = stream_context_create($opts);
    $result = file_get_contents($url, false, $context);
    $decoded = json_decode($result, true);

    logToFile(['decreaseStock' => ['variationId' => $variationId, 'quantity' => $quantity, 'response' => $decoded]]);
    
    return $decoded;
}
function processPayment($chat_id) {
    global $db;

    try {
        sendMessage($chat_id, " ✅ مشخصات شما تایید شد.");
        sendMessage($chat_id, "مبلغ 100000 تومان بابت هزینه ارسال به سبد خرید شما اضافه میشود");
        sendMessage($chat_id, "⏳ در حال انتقال به درگاه پرداخت...");

        $items = getCartItems($chat_id);
        if (empty($items)) {
            sendMessage($chat_id, "سبد خرید شما خالی است.");
            return;
        }

        $amount = 0;
        $delivery = 100000;
        foreach ($items as $it) {
            if ($it['price'] <= 0) {
                sendMessage($chat_id, "قیمت نامعتبر برای یک آیتم؛ لطفاً دوباره تلاش کنید.");
                return;
            }
            $amount += $it['price'] * $it['quantity'];
            $amount += $delivery;
        }
        if ($amount <= 0) {
            sendMessage($chat_id, "جمع مبلغ نامعتبر است.");
            return;
        }

        $user = getUserInfo($chat_id);

        $stmt = $db->prepare("
            INSERT INTO orders (telegram_id, amount, status, gateway, user_snapshot, items_snapshot) 
            VALUES (?, ?, 'pending', 'zarinpal', ?, ?)
        ");
        $stmt->execute([
            $chat_id,
            $amount,
            json_encode($user, JSON_UNESCAPED_UNICODE),
            json_encode($items, JSON_UNESCAPED_UNICODE)
        ]);
        $orderId = $db->lastInsertId();

        $callbackUrl = BASE_URL . "bot/callback_zarinpal.php?order_id=" . $orderId;

        $zarinpal = zarinpal_request(
            ZARINPAL_MERCHANT_ID,
            $amount,
            "پرداخت سفارش شماره {$orderId}",
            $callbackUrl
        );

        if (!$zarinpal) {
            sendMessage($chat_id, "⚠️ خطا در ایجاد لینک پرداخت. لطفاً دوباره تلاش کنید.");
            $db->prepare("UPDATE orders SET status='failed' WHERE id=? AND status='pending'")->execute([$orderId]);
            return;
        }

        $db->prepare("UPDATE orders SET token=? WHERE id=?")
           ->execute([$zarinpal['authority'], $orderId]);

        $keyboard = [
            'inline_keyboard' => [
                [['text' => 'پرداخت امن 💳', 'url' => $zarinpal['link']]]
            ]
        ];
        sendMessage(
            $chat_id,
            "شماره سفارش: #$orderId\nمبلغ: " . number_format($amount) . " تومان\nروی دکمه‌ی زیر بزنید تا وارد درگاه شوید:",
            $keyboard
        );


    } catch (Throwable $e) {
        sendMessage($chat_id, "⚠️ خطای داخلی: " . $e->getMessage());
    }
}

// sendPhoto تابع جدید مشابه sendMessage
function sendPhoto($chat_id, $photo_url, $caption = null, $reply_markup = null) {
    $post_fields = [
        'chat_id' => $chat_id,
        'photo' => $photo_url,
        'parse_mode' => 'HTML' // ← اضافه شد
    ];

    if ($caption !== null) {
        $post_fields['caption'] = $caption;
    }

    if ($reply_markup !== null) {
        $post_fields['reply_markup'] = json_encode($reply_markup, JSON_UNESCAPED_UNICODE);
    }

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, "https://api.telegram.org/bot" . BOT_TOKEN . "/sendPhoto");
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $post_fields);
    $result = curl_exec($ch);
    curl_close($ch);

    return json_decode($result, true);
}
function sendMediaGroup($chat_id, $media) {
    $post_fields = [
        'chat_id' => $chat_id,
        'media' => json_encode($media)
    ];

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, "https://api.telegram.org/bot" . BOT_TOKEN . "/sendMediaGroup");
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $post_fields);
    $result = curl_exec($ch);
    curl_close($ch);

    return json_decode($result, true);
}
function showProductsByCategory($chat_id, $category_id, $page = 1) {
    logToFile("Fetching products for category ID $category_id, page $page"); // لاگ برای شروع درخواست

    $products = getProductsByCategory($category_id, $page, 10); // دریافت محصولات

    // لاگ برای داده‌های دریافتی
    logToFile("Products received for category $category_id: " . print_r($products, true));

    if ($products['Code'] == 1) {
        $keyboard = ['inline_keyboard' => []];

        foreach ($products['Result']['List'] as $p) {
            $keyboard['inline_keyboard'][] = [
                ['text' => "{$p['Name']}", 'callback_data' => "product_{$p['Id']}"]
            ];
        }

        // لاگ برای دکمه‌های صفحه‌بندی
        $nav_buttons = [];
        if ($page > 1) {
            $nav_buttons[] = ['text' => "⬅️ قبلی", 'callback_data' => "category_{$category_id}_page_" . ($page - 1)];
        }
        if ($page * 10 < $products['Result']['Count']) {
            $nav_buttons[] = ['text' => "بعدی ➡️", 'callback_data' => "category_{$category_id}_page_" . ($page + 1)];
        }

        if (!empty($nav_buttons)) {
            $keyboard['inline_keyboard'][] = $nav_buttons;
            logToFile("Pagination buttons added for category $category_id: " . print_r($nav_buttons, true)); // لاگ برای دکمه‌های صفحه‌بندی
        }

        sendMessage($chat_id, "📄 صفحه $page از محصولات دسته‌بندی:", $keyboard);
    } else {
        logToFile("Error fetching products for category $category_id: " . print_r($products, true)); // لاگ برای خطا
        sendMessage($chat_id, "مشکلی در دریافت محصولات پیش اومد!");
    }
}


// گرفتن محصولات بر اساس دسته‌بندی
function getProductsByCategory($category_id, $page = 1, $limit = 10) {
    $url = API_BASE . "Search/Products?page=$page&limit=$limit";
    $filters = ['Category' => [$category_id]]; // فیلتر بر اساس دسته‌بندی

    $opts = [
        'http' => [
            'method'  => 'POST',
            'header'  => 'Content-Type: application/json',
            'content' => json_encode($filters, JSON_UNESCAPED_UNICODE)
        ]
    ];
    $context = stream_context_create($opts);

    // لاگ برای ارسال درخواست
    logToFile("Requesting products for category ID $category_id with filters: " . json_encode($filters));

    $response = json_decode(file_get_contents($url, false, $context), true);

    // لاگ برای پاسخ دریافتی
    logToFile("Response for category $category_id: " . print_r($response, true));

    if (!$response || $response['Code'] != 1) {
        logToFile("Error fetching products for category $category_id: " . print_r($response, true));
        return null;
    }

    return $response;
}
function getCategories() {
    $url = API_BASE . "Cats";
    $response = file_get_contents($url);
    
    // لاگ برای مشاهده درخواست و پاسخ دریافتی
    logToFile("Requesting categories from API: $url");
    logToFile("Response for categories: " . $response);

    $data = json_decode($response, true);

    // بررسی وضعیت کد API
    if ($data['Code'] == 1) {
        return $data['Result'];  // در صورتی که داده‌ها درست باشند
    } else {
        logToFile("Error fetching categories: " . print_r($data, true));
        return null;  // اگر مشکلی در داده‌ها بود
    }
}
function showCategoryProducts($chat_id, $categories) {
    logToFile("Displaying categories: " . print_r($categories, true));

    if (empty($categories)) {
        sendMessage($chat_id, "❌ هیچ دسته‌بندی برای نمایش وجود ندارد.");
        return;
    }

    $keyboard = ['inline_keyboard' => []];
    foreach ($categories as $category) {
        $keyboard['inline_keyboard'][] = [
            ['text' => $category['Name'], 'callback_data' => "category_{$category['Id']}"]
        ];
    }

    sendMessage($chat_id, "لطفاً یک دسته‌بندی انتخاب کنید:", $keyboard);
}

function getCategoriesByGender($gender) {
    logToFile("Fetching categories for gender: $gender");

    $categories = getCategories(); // گرفتن تمام دسته‌بندی‌ها

    // فیلتر کردن بر اساس جنسیت
    $filteredCategories = array_filter($categories, function($category) use ($gender) {
        return $category['Gender'] == $gender;
    });

    // لاگ گرفتن دسته‌بندی‌های فیلتر شده
    logToFile("Filtered categories: " . print_r($filteredCategories, true));

    return $filteredCategories;
}
// ذخیره آیدی محصول در جدول
function logSentProduct($product_id) {
    global $db;

    $stmt = $db->prepare("INSERT INTO sent_products (id) VALUES (?)");
    $stmt->execute([$product_id]);
}
// بررسی ارسال قبلی محصول
function isProductSent($product_id) {
    global $db;

    $stmt = $db->prepare("SELECT COUNT(*) FROM sent_products WHERE id = ?");
    $stmt->execute([$product_id]);
    return $stmt->fetchColumn() > 0;
}
// حذف محصولات قدیمی تر از یک هفته
function deleteOldProducts() {
    global $db;

    $stmt = $db->prepare("DELETE FROM sent_products WHERE timestamp < NOW() - INTERVAL 1 WEEK");
    $stmt->execute();
}



